/*
 * Copyright (C) 2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_NDEBUG 0
#define LOG_TAG "AACExtractor"
#include <utils/Log.h>

#include "include/AACExtractor.h"
#include "include/avc_utils.h"

#include <media/stagefright/foundation/ABuffer.h>
#include <media/stagefright/DataSource.h>
#include <media/stagefright/MediaBufferGroup.h>
#include <media/stagefright/MediaDebug.h>
#include <media/stagefright/MediaDefs.h>
#include <media/stagefright/MediaErrors.h>
#include <media/stagefright/MediaSource.h>
#include <media/stagefright/MetaData.h>
#include <utils/String8.h>

namespace android {

#define ADTS_HEADER_LENGTH 7

class AACSource : public MediaSource {
public:
    AACSource(const sp<DataSource> &source,
              const sp<MetaData> &meta,
              const Vector<uint64_t> &offset_vector,
              int64_t frame_duration_us,
              size_t firstFramesize);

    virtual status_t start(MetaData *params = NULL);
    virtual status_t stop();

    virtual sp<MetaData> getFormat();

    virtual status_t read(
            MediaBuffer **buffer, const ReadOptions *options = NULL);

protected:
    virtual ~AACSource();

private:
    static const size_t kMaxFrameSize;
    sp<DataSource> mDataSource;
    sp<MetaData> mMeta;

    off64_t mOffset;
    int64_t mCurrentTimeUs;
    bool mStarted;
    MediaBufferGroup *mGroup;

    Vector<uint64_t> mOffsetVector;
    int64_t mFrameDurationUs;
    bool isFirstframe;
    int64_t frameOffset;
    size_t first_framesize;	

    AACSource(const AACSource &);
    AACSource &operator=(const AACSource &);
};

////////////////////////////////////////////////////////////////////////////////

// Returns the sample rate based on the sampling frequency index
uint32_t get_sample_rate(const uint8_t sf_index)
{
    static const uint32_t sample_rates[] =
    {
        96000, 88200, 64000, 48000, 44100, 32000,
        24000, 22050, 16000, 12000, 11025, 8000
    };

    if (sf_index < sizeof(sample_rates) / sizeof(sample_rates[0])) {
        return sample_rates[sf_index];
    }

    return 0;
}

static size_t getFrameSize(const sp<DataSource> &source, off64_t offset) {
    size_t frameSize = 0;

    uint8_t syncword[2];
    if (source->readAt(offset, &syncword, 2) != 2) {
        return 0;
    }
    if ((syncword[0] != 0xff) || ((syncword[1] & 0xf6) != 0xf0)) {
        return 0;
    }

    uint8_t protectionAbsent;
    protectionAbsent = syncword[1];	
    /*if (source->readAt(offset + 1, &protectionAbsent, 1) < 1) {
        return 0;
    }*/
    protectionAbsent &= 0x1;
    uint8_t dummy;
    if (source->readAt(offset + 2, &dummy, 1) != 1) {
        return 0;
    }	

    uint8_t header[3];
    if (source->readAt(offset + 3, &header, 3) < 3) {
        return 0;
    }

    frameSize = (header[0] & 0x3) << 11 | header[1] << 3 | header[2] >> 5;
    frameSize += ADTS_HEADER_LENGTH + protectionAbsent ? 0 : 2;
    if (source->readAt(offset+6, &dummy, 1) != 1) {
        return 0;
    }	

    return frameSize;
}

AACExtractor::AACExtractor(const sp<DataSource> &source)
    : mDataSource(source),
      mInitCheck(NO_INIT),
      mFrameDurationUs(0) {
    String8 mimeType;
    float confidence;
    off64_t offset = 0;	
    if (!SniffAAC(mDataSource, &mimeType, &confidence, NULL)) {
        return;
    }

#if 0
    if (source->readAt(offset + 4, &header, 2) < 2) {
        return;
    }
    uint8_t header2[3];
    header2[0] = header[1];
    header2[1]=  header1[0];
    header2[2]=  header1[1];

    frameSize = (header2[0] & 0x3) << 11 | header2[1] << 3 | header2[2] >> 5;
    frameSize += ADTS_HEADER_LENGTH + 1? 0 : 2;
    LOGE("Frame size out is % d",frameSize);	
    mOffsetVector.push(offset);

    offset += frameSize;
    numFrames ++;	


    if (mDataSource->getSize(&streamSize) == OK) {
	  LOGE("Inside get frame loop");	
         while (offset < streamSize) {
            if ((frameSize = getFrameSize(source, offset)) == 0) {
                return;
            }
	     LOGE("Frame size in Extractor is %d",frameSize);	
            mOffsetVector.push(offset);

            offset += frameSize;
            numFrames ++;
        }

        // Round up and get the duration
        mFrameDurationUs = (1024 * 1000000ll + (sr - 1)) / sr;
        duration = numFrames * mFrameDurationUs;
        mMeta->setInt64(kKeyDuration, duration);
    }
#endif
    uint8_t profile, sf_index, channel, header[2];
    if (mDataSource->readAt(offset+ 2, &header, 2) < 2) {
        return;
    }
    profile = (header[0] >> 6) & 0x3;
    sf_index = (header[0] >> 2) & 0xf;
    uint32_t sr = get_sample_rate(sf_index);
    if (sr == 0) {
            return ;
    }   
    channel = (header[0] & 0x1) << 2 | (header[1] >> 6);
    LOGE("Profile : %d :: SF: %d :: channel : %d",profile,sr,channel);	

    mMeta = MakeAACCodecSpecificData(profile, sf_index, channel);
    mFrameDurationUs = (1024 * 1000000ll + (sr - 1)) / sr;

    off64_t streamSize, numFrames = 0;
    size_t frameSize = 0;
    int64_t duration = 0;
    uint8_t protectionAbsent;
 
    protectionAbsent = 0x1;	
    /*if (source->readAt(offset + 1, &protectionAbsent, 1) < 1) {
        return 0;
    }*/

    uint8_t header1[3];
    if (mDataSource->readAt(offset + 4, &header1, 2) < 2) {
        return;
    }
    
    frameSize = (header[1] & 0x3) << 11 | header1[0] << 3 | header1[1] >> 5;
    frameSize += ADTS_HEADER_LENGTH + protectionAbsent ? 0 : 2;
    LOGE("First Frame size is %d",frameSize);
    firstFramesize = frameSize;	

    if (mDataSource->readAt(offset+ 6, &header, 1) < 1) {
        return;
    }	

    mInitCheck = OK;
}

AACExtractor::~AACExtractor() {
}

sp<MetaData> AACExtractor::getMetaData() {
    sp<MetaData> meta = new MetaData;

    if (mInitCheck != OK) {
        return meta;
    }

    meta->setCString(kKeyMIMEType, MEDIA_MIMETYPE_AUDIO_AAC);

    return meta;
}

size_t AACExtractor::countTracks() {
    return mInitCheck == OK ? 1 : 0;
}

sp<MediaSource> AACExtractor::getTrack(size_t index) {
    if (mInitCheck != OK || index != 0) {
        return NULL;
    }

    return new AACSource(mDataSource, mMeta, mOffsetVector, mFrameDurationUs,firstFramesize);
}

sp<MetaData> AACExtractor::getTrackMetaData(size_t index, uint32_t flags) {
    if (mInitCheck != OK || index != 0) {
        return NULL;
    }

    return mMeta;
}

////////////////////////////////////////////////////////////////////////////////

// 8192 = 2^13, 13bit AAC frame size (in bytes)
const size_t AACSource::kMaxFrameSize = 8192;

AACSource::AACSource(
        const sp<DataSource> &source, const sp<MetaData> &meta,
        const Vector<uint64_t> &offset_vector,
        int64_t frame_duration_us,
        size_t firstFramesize)
    : mDataSource(source),
      mMeta(meta),
      mOffset(0),
      mCurrentTimeUs(0),
      mStarted(false),
      mGroup(NULL),
      mOffsetVector(offset_vector),
      mFrameDurationUs(frame_duration_us),
      isFirstframe(true),
      frameOffset(0),
      first_framesize(firstFramesize){
}

AACSource::~AACSource() {
    if (mStarted) {
        stop();
    }
}

status_t AACSource::start(MetaData *params) {
    CHECK(!mStarted);
    mOffset = 0;
    mCurrentTimeUs = 0;
    mGroup = new MediaBufferGroup;
    mGroup->add_buffer(new MediaBuffer(kMaxFrameSize));
    mStarted = true;

    return OK;
}

status_t AACSource::stop() {
    CHECK(mStarted);

    delete mGroup;
    mGroup = NULL;

    mStarted = false;
    return OK;
}

sp<MetaData> AACSource::getFormat() {
    return mMeta;
}

status_t AACSource::read(
        MediaBuffer **out, const ReadOptions *options) {
    *out = NULL;
    size_t frameSize, frameSizeWithoutHeader;	
    int64_t seekTimeUs;
    if(isFirstframe) {
	 frameSize = first_framesize;
	 isFirstframe = false;
        MediaBuffer *buffer = new MediaBuffer(kMaxFrameSize) ;
        /*status_t err = mGroup->acquire_buffer(&buffer);
        if (err != OK) {
            return err;
        }*/
    
        frameSizeWithoutHeader = frameSize - ADTS_HEADER_LENGTH;
        LOGV("Frame size without header is %d",frameSizeWithoutHeader);	
        if (mDataSource->readAt(mOffset + ADTS_HEADER_LENGTH, buffer->data(),
                    frameSizeWithoutHeader) != (ssize_t)frameSizeWithoutHeader) {
            buffer->release();
            buffer = NULL;
	     LOGE("Unable to read from Live Source"); 	

            return ERROR_IO;
        }

        LOGV ("AACSource::read(), kKeyTime: %lld", mCurrentTimeUs);

        buffer->set_range(0, frameSizeWithoutHeader);
        buffer->meta_data()->setInt64(kKeyTime, mCurrentTimeUs);
        buffer->meta_data()->setInt32(kKeyIsSyncFrame, 1);

        mOffset += frameSize;
        mCurrentTimeUs += mFrameDurationUs;

        *out = buffer;
        return OK;	 	        
    }
	
    ReadOptions::SeekMode mode;
    if (options && options->getSeekTo(&seekTimeUs, &mode)) {
        if (mFrameDurationUs > 0) {
            int64_t seekFrame = seekTimeUs / mFrameDurationUs;
            mCurrentTimeUs = seekFrame * mFrameDurationUs;

            mOffset = mOffsetVector.itemAt(seekFrame);
        }
    }
    LOGV("offset AAC is %lld",mOffset);
   
    if ((frameSize = getFrameSize(mDataSource, mOffset)) == 0) {
        return ERROR_END_OF_STREAM;
    }
    LOGV("Frame size is %d",frameSize);

    /*MediaBuffer *buffer;
    status_t err = mGroup->acquire_buffer(&buffer);
    if (err != OK) {
	 LOGE("Aquire mgroup buffer fail");	
        return err;
    }*/
    
    MediaBuffer *buffer = new MediaBuffer(kMaxFrameSize) ;
	
    frameSizeWithoutHeader = frameSize - ADTS_HEADER_LENGTH;
    LOGV("Frame size without header is %d",frameSizeWithoutHeader);	
    if (mDataSource->readAt(mOffset + ADTS_HEADER_LENGTH, buffer->data(),
                frameSizeWithoutHeader) != (ssize_t)frameSizeWithoutHeader) {
        buffer->release();
        buffer = NULL;
	 LOGE("Unable to read from Live Source EOS reached"); 	

        return ERROR_IO;
    }

    LOGE ("AACSource::read(), kKeyTime: %lld", mCurrentTimeUs);

    buffer->set_range(0, frameSizeWithoutHeader);
    buffer->meta_data()->setInt64(kKeyTime, mCurrentTimeUs);
    buffer->meta_data()->setInt32(kKeyIsSyncFrame, 1);

    mOffset += frameSize;
    mCurrentTimeUs += mFrameDurationUs;

    *out = buffer;
    return OK;
}

////////////////////////////////////////////////////////////////////////////////

bool SniffAAC(
        const sp<DataSource> &source, String8 *mimeType, float *confidence,
        sp<AMessage> *) {
    uint8_t header[2];
 
    if (source->readAt(0, &header, 2) != 2) {
        return false;
    }

    // ADTS syncword
    if ((header[0] == 0xff) && ((header[1] & 0xf6) == 0xf0)) {
        *mimeType = MEDIA_MIMETYPE_AUDIO_AAC_ADTS;
        *confidence = 0.2;
	 LOGE("MIME TYPE AAC..");	
        return true;
    }

    return false;
}

}  // namespace android
