/*
 * Copyright (C) 2010 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//#define LOG_NDEBUG 0
#define LOG_TAG "LiveSession"
#include <utils/Log.h>

#include "include/LiveSession.h"

#include "LiveDataSource.h"

#include "include/M3UParser.h"
#include "include/NuHTTPDataSource.h"

#include <cutils/properties.h>
#include <media/stagefright/foundation/hexdump.h>
#include <media/stagefright/foundation/ABuffer.h>
#include <media/stagefright/foundation/ADebug.h>
#include <media/stagefright/foundation/AMessage.h>
#include <media/stagefright/DataSource.h>
#include <media/stagefright/FileSource.h>
#include <media/stagefright/MediaErrors.h>

#include <ctype.h>
#include <openssl/aes.h>
#define AAC_SUPPORT 1

#ifdef SEC_MMFW_HLS
#include "include/avc_utils.h"
#include <media/stagefright/foundation/ABitReader.h>
#include <media/stagefright/MediaDefs.h>
#include <media/stagefright/MetaData.h>
#include <utils/KeyedVector.h>
#endif
namespace android {

const int64_t LiveSession::kMaxPlaylistAgeUs = 15000000ll;
#ifdef SEC_MMFW_HLS
const size_t kHLSLowWaterMarkBytes = 150000;//40000
const size_t kHLSHighWaterMarkBytes = 700000;//200000
#endif

LiveSession::LiveSession(uint32_t flags)
    : mFlags(flags),
      mDataSource(new LiveDataSource),
      mHTTPDataSource(
              new NuHTTPDataSource(
                  (mFlags & kFlagIncognito)
                    ? NuHTTPDataSource::kFlagIncognito
                    : 0)),
      mPrevBandwidthIndex(-1),
      mLastPlaylistFetchTimeUs(-1),
#ifdef SEC_MMFW_HLS
      mLastMediaFileDurationSec(0), /* WONPIA initial reload delay */
      mDownloadMasterlist(true),
	  mDownloadPlaylistStartTime(0),
	  mDownloadPlaylistEndTime(0),
	  mDownloadDeltaTime(0),
	  mPlayTime(0),
	  mPrevHeight(0),
	  mPrevWidth(0),
	  mResolution(false),
	  mSingleBandwidth(false),
	  mAudioOnly(false),
	  mIndex(0),
	  mPrevIndex(0),
      mNumRetriesTSconnection(0),
	  mNumRetriesTScontent(0),  /* WONPIA 20110317 recv Content Error(timeout, etc) */
	  mCheckFirstBW(false),
#endif
      mSeqNumber(-1),
      mSeekTimeUs(-1),
      mNumRetries(0),
      mDurationUs(-1),
      mSeekDone(false),
      mDisconnectPending(false),
      mMonitorQueueGeneration(0) {
}

LiveSession::~LiveSession() {
}

sp<DataSource> LiveSession::getDataSource() {
    return mDataSource;
}

void LiveSession::connect(const char *url) {
    sp<AMessage> msg = new AMessage(kWhatConnect, id());
    msg->setString("url", url);
    msg->post();
}

void LiveSession::disconnect() {
    Mutex::Autolock autoLock(mLock);
    mDisconnectPending = true;

    mHTTPDataSource->disconnect();

    (new AMessage(kWhatDisconnect, id()))->post();
}

void LiveSession::seekTo(int64_t timeUs) {
    Mutex::Autolock autoLock(mLock);
    mSeekDone = false;

    sp<AMessage> msg = new AMessage(kWhatSeek, id());
    msg->setInt64("timeUs", timeUs);
    msg->post();

    while (!mSeekDone) {
        mCondition.wait(mLock);
    }
}

void LiveSession::onMessageReceived(const sp<AMessage> &msg) {
    switch (msg->what()) {
        case kWhatConnect:
            onConnect(msg);
            break;

        case kWhatDisconnect:
            onDisconnect();
            break;

        case kWhatMonitorQueue:
        {
            int32_t generation;
            CHECK(msg->findInt32("generation", &generation));

            if (generation != mMonitorQueueGeneration) {
                // Stale event
                break;
            }

            onMonitorQueue();
            break;
        }

        case kWhatSeek:
            onSeek(msg);
            break;

        default:
            TRESPASS();
            break;
    }
}

// static
int LiveSession::SortByBandwidth(const BandwidthItem *a, const BandwidthItem *b) {
    if (a->mBandwidth < b->mBandwidth) {
        return -1;
    } else if (a->mBandwidth == b->mBandwidth) {
        return 0;
    }

    return 1;
}

void LiveSession::onConnect(const sp<AMessage> &msg) {
    AString url;
    CHECK(msg->findString("url", &url));

    if (!(mFlags & kFlagIncognito)) {
        LOGI("onConnect '%s'", url.c_str());
    } else {
        LOGI("onConnect <URL suppressed>");
    }

    mMasterURL = url;

    sp<M3UParser> playlist = fetchPlaylist(url.c_str());

    if (playlist == NULL) {
        LOGE("unable to fetch master playlist '%s'.", url.c_str());

        mDataSource->queueEOS(ERROR_IO);
        return;
    }

    if (playlist->isVariantPlaylist()) {
        for (size_t i = 0; i < playlist->size(); ++i) {
            BandwidthItem item;

            sp<AMessage> meta;
            playlist->itemAt(i, &item.mURI, &meta);

            unsigned long bandwidth;
            CHECK(meta->findInt32("bandwidth", (int32_t *)&item.mBandwidth));

            mBandwidthItems.push(item);
        }

        CHECK_GT(mBandwidthItems.size(), 0u);

        mBandwidthItems.sort(SortByBandwidth);
    }
#ifdef SEC_MMFW_HLS
	else
	{
		mPlaylistWithoutSub = playlist;
	}
#endif
    postMonitorQueue();
}

void LiveSession::onDisconnect() {
    LOGI("onDisconnect");

    mDataSource->queueEOS(ERROR_END_OF_STREAM);

    Mutex::Autolock autoLock(mLock);
    mDisconnectPending = false;
}

status_t LiveSession::fetchFile(const char *url, sp<ABuffer> *out) {
    *out = NULL;

    sp<DataSource> source;

    if (!strncasecmp(url, "file://", 7)) {
        source = new FileSource(url + 7);
    } else if (strncasecmp(url, "http://", 7)
            && strncasecmp(url, "https://", 8)) {
        return ERROR_UNSUPPORTED;
    } else {
        {
            Mutex::Autolock autoLock(mLock);

            if (mDisconnectPending) {
                return ERROR_IO;
            }
        }

#ifdef SEC_MMFW_HTTP_ENHANCEMENT
	if( strcasestr(url, "m3u8") != NULL ){ /* index file */
LOGE("fetchFile - playlist set HTTP retry YES");		
		mHTTPDataSource->setRetrialPolicy(NuHTTPDataSource::CONN_RETRY, 20, 5*1000, 300*1000); /* retry count, 5 sec interval , totoal timeout*/
        mHTTPDataSource->setRetrialPolicy(NuHTTPDataSource::READ_RETRY, 20, 5*1000, 300*1000);  /* retry count, 5 sec interval , totoal timeout*/
	}
	else{ /* ts file */
LOGE("fetchFile - ts set HTTP retry NO");				
		mHTTPDataSource->setRetrialPolicy(NuHTTPDataSource::CONN_RETRY, 0, 0, 300*1000);    /* timout must be more than '0' */
			
        mHTTPDataSource->setRetrialPolicy(NuHTTPDataSource::READ_RETRY, 0, 0, 300*1000);
	}
#endif
        status_t err = mHTTPDataSource->connect(url);
		LOGV("fetchfile = %s", url);

#ifdef SEC_MMFW_HLS
		if (err == 404) {
        	LOGE("fetchFile 404 Not Found");
            return err;
		}else if(err != OK){
        	LOGE("fetchFile Not OK by mHTTPDataSource->connect - %d : %s", err, strerror(err));
			return -1;
		}
#else
        if (err != OK) {
            return err;
        }
#endif

        source = mHTTPDataSource;
    }

    off64_t size;
    status_t err = source->getSize(&size);
    LOGE("SIZE of FILE is %lld",size);	

    if (err != OK) {
        size = 65536;
    }

    sp<ABuffer> buffer = new ABuffer(size);
    buffer->setRange(0, 0);

    for (;;) {
        size_t bufferRemaining = buffer->capacity() - buffer->size();

        if (bufferRemaining == 0) {
            bufferRemaining = 32768;

            LOGV("increasing download buffer to %d bytes",
                 buffer->size() + bufferRemaining);

            sp<ABuffer> copy = new ABuffer(buffer->size() + bufferRemaining);
            memcpy(copy->data(), buffer->data(), buffer->size());
            copy->setRange(0, buffer->size());

            buffer = copy;
        }

        ssize_t n = source->readAt(
                buffer->size(), buffer->data() + buffer->size(),
                bufferRemaining);

#ifdef SEC_MMFW_HLS
		if (n == 404) {
        	LOGE("fetchFile 404 Not Found");
            return n;
		}else if(n != OK && n < 0){
        	LOGE("fetchFile Not OK by mHTTPDataSource->readAt - %d : %s", n, strerror(n));
			return -2;
		}
#else
        if (n < 0) {
            return n;
        }
#endif

        if (n == 0) {
            break;
        }

        buffer->setRange(0, buffer->size() + (size_t)n);
    }

    *out = buffer;

    return OK;
}

sp<M3UParser> LiveSession::fetchPlaylist(const char *url) {
    sp<ABuffer> buffer;
#ifdef SEC_MMFW_HLS
	mPlayTime = 0;
	mDownloadPlaylistStartTime = ALooper::GetNowUs();
#endif
    status_t err = fetchFile(url, &buffer);

    if (err != OK) {
        return NULL;
    }

    sp<M3UParser> playlist =
        new M3UParser(url, buffer->data(), buffer->size());

    if (playlist->initCheck() != OK) {
        return NULL;
    }

    return playlist;
}

static double uniformRand() {
    return (double)rand() / RAND_MAX;
}

size_t LiveSession::getBandwidthIndex() {
#ifdef SEC_MMFW_HLS
	//explain : mBandwidthItems.size() = 0 --> No SubPlaylist
	//		  : mBandwidthItems.size() = 1 --> just 1 bandwidth in MasterPlaylist
    if (mBandwidthItems.size() == 0 || mBandwidthItems.size() == 1) {
		LOGE("mBandwidthItems.size() is %d", mBandwidthItems.size() );
		mDownloadMasterlist = false;
		mSingleBandwidth = true;
#else
    if (mBandwidthItems.size() == 0) {
#endif
        return 0;
    }
#ifdef SEC_MMFW_HLS
/*	if(mAudioOnly){
		//mAudioOnly = false;
		//mDownloadMasterlist = false;
		LOGE("retry getBandwidthIndex, because of Audio only bandwidth");
		mIndex = 0;
		return mIndex;
	}
*/
//	if(mDownloadMasterlist){
	LOGE("First TS Down mBandwidthItems.size() = %d", mBandwidthItems.size());
	int32_t bandwidthBps;
	if (mHTTPDataSource != NULL && mHTTPDataSource->estimateBandwidth(&bandwidthBps)) {
		LOGE("bandwidth estimated at %.2f kbps", bandwidthBps / 1024.0f);
    } else {
		LOGE("no bandwidth estimate.");
		mIndex = 0;
	    return mIndex;  // Pick the lowest bandwidth stream by default.
	}

	char value[PROPERTY_VALUE_MAX];
	if (property_get("media.httplive.max-bw", value, NULL)) {
		char *end;
	    long maxBw = strtoul(value, &end, 10);
	    if (end > value && *end == '\0') {
	    	if (maxBw > 0 && bandwidthBps > maxBw) {
	        	LOGE("bandwidth capped to %ld bps", maxBw);
				bandwidthBps = maxBw;
			}
	    }
	}

    // Consider only 80% of the available bandwidth usable.
    bandwidthBps = (bandwidthBps * 8) / 10;

    // Pick the highest bandwidth stream below or equal to estimated bandwidth.
    mIndex = mBandwidthItems.size() - 1;
    while (mIndex > 0 && mBandwidthItems.itemAt(mIndex).mBandwidth
                            > (size_t)bandwidthBps) {
        --mIndex;
    }
	LOGE("mBandwidth[%d] = %d, bandwidthBps  = %d", mIndex, mBandwidthItems.itemAt(mIndex).mBandwidth, bandwidthBps);
	LOGE("Num of Buffered TS = %d, Pre-Index  = %d, Cur-Index = %d",  mDataSource->countQueuedBuffers(),mPrevBandwidthIndex, mIndex);

	LOGE("selected mIndex = %d", mIndex);
	return mIndex;
//	}
/*	else{
		mPrevIndex = mIndex;
		LOGE("Next TS Down mPrevIndex = %d, mIndex = %d, mBandwidthItems.size() = %d", mPrevIndex, mIndex, mBandwidthItems.size());

		//check the Dealta Time FetchPlaylist to FetchFile
		mDownloadDeltaTime = mDownloadPlaylistEndTime - mDownloadPlaylistStartTime;
		LOGE("Check BufferSize = %lld, Delta time = %lld, mPlayTime = %d",mDataSource->getSizeQueuedData(), mDownloadDeltaTime/1000000ll, mPlayTime);

		if( mDataSource->getSizeQueuedData() < kHLSLowWaterMarkBytes || (mPlayTime * 0.75) <= (mDownloadDeltaTime / 1000000ll)){
			if(mIndex == 0){
				LOGE("Already Lowest Bandwidth ");
			}
			else{
				LOGE("Down Bandwidth ");
				mIndex--;
			}
		}
		else if(mDataSource->getSizeQueuedData() >= kHLSHighWaterMarkBytes || (mPlayTime * 0.75) > (mDownloadDeltaTime / 1000000ll)){
				if(mIndex == mBandwidthItems.size() - 1){
					LOGE("Already Highest Bandwidth");
				}
				else{
					LOGE("UP Bandwidth ");
					mIndex++;
				}
		}
//Reject the Resolution Change
//Don't need in P4
		if(mPrevIndex != mIndex){
			LOGE("Current Index[%d] : width = %d, height = %d", mIndex, mBandwidthItems.editItemAt(mIndex).mBandwidthWidth, mBandwidthItems.editItemAt(mIndex).mBandwidthHeight);
			if(mBandwidthItems.editItemAt(mIndex).mBandwidthWidth != 0 && mBandwidthItems.editItemAt(mIndex).mBandwidthHeight != 0){
				LOGE("Prev Index[%d] width = %d, height = %d", mPrevIndex, mBandwidthItems.editItemAt(mPrevIndex).mBandwidthWidth, mBandwidthItems.editItemAt(mPrevIndex).mBandwidthHeight);
				if(mBandwidthItems.editItemAt(mPrevIndex).mBandwidthWidth != mBandwidthItems.editItemAt(mIndex).mBandwidthWidth || mBandwidthItems.editItemAt(mPrevIndex).mBandwidthHeight != mBandwidthItems.editItemAt(mIndex).mBandwidthHeight){
					LOGE("Request mIndex Already reject, because if Resolution");
					mIndex = mPrevIndex;
				}
			}
		}
		LOGE("selected mIndex = %d", mIndex);
		return mIndex;
	}*/
#else
#if 1
    int32_t bandwidthBps;
    if (mHTTPDataSource != NULL
            && mHTTPDataSource->estimateBandwidth(&bandwidthBps)) {
        LOGV("bandwidth estimated at %.2f kbps", bandwidthBps / 1024.0f);
    } else {
        LOGV("no bandwidth estimate.");
        return 0;  // Pick the lowest bandwidth stream by default.
    }

    char value[PROPERTY_VALUE_MAX];
    if (property_get("media.httplive.max-bw", value, NULL)) {
        char *end;
        long maxBw = strtoul(value, &end, 10);
        if (end > value && *end == '\0') {
            if (maxBw > 0 && bandwidthBps > maxBw) {
                LOGV("bandwidth capped to %ld bps", maxBw);
                bandwidthBps = maxBw;
            }
        }
    }

    // Consider only 80% of the available bandwidth usable.
    bandwidthBps = (bandwidthBps * 8) / 10;

    // Pick the highest bandwidth stream below or equal to estimated bandwidth.

    size_t index = mBandwidthItems.size() - 1;
    while (index > 0 && mBandwidthItems.itemAt(index).mBandwidth
                            > (size_t)bandwidthBps) {
        --index;
    }
#elif 0
    // Change bandwidth at random()
    size_t index = uniformRand() * mBandwidthItems.size();
#elif 0
    // There's a 50% chance to stay on the current bandwidth and
    // a 50% chance to switch to the next higher bandwidth (wrapping around
    // to lowest)
    const size_t kMinIndex = 0;

    size_t index;
    if (mPrevBandwidthIndex < 0) {
        index = kMinIndex;
    } else if (uniformRand() < 0.5) {
        index = (size_t)mPrevBandwidthIndex;
    } else {
        index = mPrevBandwidthIndex + 1;
        if (index == mBandwidthItems.size()) {
            index = kMinIndex;
        }
    }
#elif 0
    // Pick the highest bandwidth stream below or equal to 1.2 Mbit/sec

    size_t index = mBandwidthItems.size() - 1;
    while (index > 0 && mBandwidthItems.itemAt(index).mBandwidth > 1200000) {
        --index;
    }
#else
    size_t index = mBandwidthItems.size() - 1;  // Highest bandwidth stream
#endif
    return index;
#endif
}

void LiveSession::onDownloadNext() {
    size_t bandwidthIndex = getBandwidthIndex();

rinse_repeat:
    int64_t nowUs = ALooper::GetNowUs();

#ifdef SEC_MMFW_HLS
static int a = 0;
LOGE("%d - playlist load start : mLastPlaylistFetchTimeUs - %lld, bandwidthIndex - %d, mPrevBandwidthIndex - %d", a, mLastPlaylistFetchTimeUs, bandwidthIndex, mPrevBandwidthIndex);
#endif
    if (mLastPlaylistFetchTimeUs < 0
            || (ssize_t)bandwidthIndex != mPrevBandwidthIndex
            || (!mPlaylist->isComplete()
                && mLastPlaylistFetchTimeUs + kMaxPlaylistAgeUs <= nowUs)) {
        AString url;
/*		
#ifdef SEC_MMFW_HLS
		if(mResolution)
		{
			LOGE("Use the Prev Bandwidth bacause of Resolution");
			bandwidthIndex = mPrevBandwidthIndex;
			mIndex = bandwidthIndex;
			mResolution = false;
		}
#endif
*/
        if (mBandwidthItems.size() > 0) {
            url = mBandwidthItems.editItemAt(bandwidthIndex).mURI;
        } else {
            url = mMasterURL;
        }

        bool firstTime = (mPlaylist == NULL);

#ifdef SEC_MMFW_HLS
		int64_t initialReloadDelayLeftUs = (int64_t)(mLastMediaFileDurationSec)*1000000ll - 3000000ll;
        // Fetch the playlist. 
		if( firstTime && (url == mMasterURL) )
		{
			mPlaylist = mPlaylistWithoutSub;
		}
	    else
		{
	        mPlaylist = fetchPlaylist(url.c_str());
        }
#else
        mPlaylist = fetchPlaylist(url.c_str());
#endif
        if (mPlaylist == NULL) {
            LOGE("failed to load playlist at url '%s'", url.c_str());
            mDataSource->queueEOS(ERROR_IO);
            return;
        }

        if (firstTime) {
            Mutex::Autolock autoLock(mLock);

            int32_t targetDuration;
            if (!mPlaylist->isComplete()
                    || !mPlaylist->meta()->findInt32(
                    "target-duration", &targetDuration)) {
                mDurationUs = -1;
            } else {
#ifdef SEC_MMFW_HLS
                mDurationUs = 1000000ll * mPlaylist->getM3U8Duration();
				LOGE("Target mDurationUs = %ld", mDurationUs);
#else
                mDurationUs = 1000000ll * targetDuration * mPlaylist->size();
#endif
            }
        }

        mLastPlaylistFetchTimeUs = ALooper::GetNowUs();
    }

    int32_t firstSeqNumberInPlaylist;
    if (mPlaylist->meta() == NULL || !mPlaylist->meta()->findInt32(
                "media-sequence", &firstSeqNumberInPlaylist)) {
        firstSeqNumberInPlaylist = 0;
    }

    bool explicitDiscontinuity = false;
    bool bandwidthChanged = false;

    if (mSeekTimeUs >= 0) {
        int32_t targetDuration;
        if (mPlaylist->isComplete() &&
                mPlaylist->meta()->findInt32(
                    "target-duration", &targetDuration)) {
            int64_t seekTimeSecs = (mSeekTimeUs + 500000ll) / 1000000ll;
            int64_t index = seekTimeSecs / targetDuration;

            if (index >= 0 && index < mPlaylist->size()) {
                int32_t newSeqNumber = firstSeqNumberInPlaylist + index;

                if (newSeqNumber != mSeqNumber) {
                    LOGI("seeking to seq no %d", newSeqNumber);

                    mSeqNumber = newSeqNumber;

                    mDataSource->reset();

                    // reseting the data source will have had the
                    // side effect of discarding any previously queued
                    // bandwidth change discontinuity.
                    // Therefore we'll need to treat these explicit
                    // discontinuities as involving a bandwidth change
                    // even if they aren't directly.
                    explicitDiscontinuity = true;
                    bandwidthChanged = true;
                }
            }
        }

        mSeekTimeUs = -1;

        Mutex::Autolock autoLock(mLock);
        mSeekDone = true;
        mCondition.broadcast();
    }

    if (mSeqNumber < 0) {
        if (mPlaylist->isComplete()) {
            mSeqNumber = firstSeqNumberInPlaylist;
        } else {
            mSeqNumber = firstSeqNumberInPlaylist + mPlaylist->size() / 2;
        }
    }

    int32_t lastSeqNumberInPlaylist =
        firstSeqNumberInPlaylist + (int32_t)mPlaylist->size() - 1;

    if (mSeqNumber < firstSeqNumberInPlaylist
            || mSeqNumber > lastSeqNumberInPlaylist) {
#ifdef SEC_MMFW_HLS
		if(mPrevBandwidthIndex > bandwidthIndex){
			LOGE("Bandwidth Down");
		}	
        else
#endif
        if (mPrevBandwidthIndex != (ssize_t)bandwidthIndex) {
            // Go back to the previous bandwidth.

            LOGI("new bandwidth does not have the sequence number "
                 "we're looking for, switching back to previous bandwidth");

            mLastPlaylistFetchTimeUs = -1;
#ifdef SEC_MMFW_HLS
			if(mPrevBandwidthIndex !=-1)
			{
		    	bandwidthIndex = mPrevBandwidthIndex;
			}
			else
			{
				mPrevBandwidthIndex = bandwidthIndex;
			}
#else
            bandwidthIndex = mPrevBandwidthIndex;
#endif
            goto rinse_repeat;
        }

        if (!mPlaylist->isComplete()
                && mSeqNumber > lastSeqNumberInPlaylist
                && mNumRetries < kMaxNumRetries) {
            ++mNumRetries;

            mLastPlaylistFetchTimeUs = -1;
            postMonitorQueue(3000000ll);
            return;
        }

        LOGE("Cannot find sequence number %d in playlist "
             "(contains %d - %d)",
             mSeqNumber, firstSeqNumberInPlaylist,
             firstSeqNumberInPlaylist + mPlaylist->size() - 1);

#ifdef SEC_MMFW_HLS
        if (!mPlaylist->isComplete()
                && mSeqNumber < firstSeqNumberInPlaylist  ) {

			mSeqNumber = firstSeqNumberInPlaylist;

		}
		else{
			// End of Stream. 
        mDataSource->queueEOS(ERROR_END_OF_STREAM);
        return;
    }
#else
        mDataSource->queueEOS(ERROR_END_OF_STREAM);
        return;
#endif				
    }

    mNumRetries = 0;

    AString uri;
    sp<AMessage> itemMeta;
    CHECK(mPlaylist->itemAt(
                mSeqNumber - firstSeqNumberInPlaylist,
                &uri,
                &itemMeta));

    int32_t val;
    if (itemMeta->findInt32("discontinuity", &val) && val != 0) {
        explicitDiscontinuity = true;
    }

#ifdef SEC_MMFW_HLS
	// Get the Target Duration of each TS streams.  
	int32_t tmpTargetDuration = 8;
    if (!mPlaylist->meta()->findInt32("target-duration", &tmpTargetDuration))
    {
		LOGW("PLAYLIST has no #EXT-X-TARGETDURATION!!!!!!");
    }
	
	// Duration.
    if (!itemMeta->findInt32("duration", &mLastMediaFileDurationSec)) {
		mLastMediaFileDurationSec = tmpTargetDuration;
    }
	mPlayTime += tmpTargetDuration;
#endif
    sp<ABuffer> buffer;
    status_t err = fetchFile(uri.c_str(), &buffer);
#ifdef SEC_MMFW_HLS
	mDownloadPlaylistEndTime = ALooper::GetNowUs();
#endif
    if (err != OK) {
        LOGE("failed to fetch .ts segment at url '%s'", uri.c_str());
#ifdef SEC_MMFW_HLS
		if(err == 404) { /* WONPIA 404 NOT FOUND */
			LOGE("onDownloadNext - TS 404 NOT FOUND");
		    ++mSeqNumber;
			mLastPlaylistFetchTimeUs = -1;
	        goto rinse_repeat;

		}else if(err == -1) { /* WONPIA Connection Error(timeout, etc) */
			if( mNumRetriesTSconnection < kMaxNumRetries ) {
				LOGE("onDownloadNext - TS Connection Error");
            	++mNumRetriesTSconnection;

				mLastPlaylistFetchTimeUs = -1;
				goto rinse_repeat;
			}else {
        mDataSource->queueEOS(err);
			}
		}else if(err == -2){ /* WONPIA 20110317 recv Content Error(timeout, etc) */
			if( mNumRetriesTScontent < kMaxNumRetries ) {
				LOGE("mNumRetriesTScontent : %d", mNumRetriesTScontent);
            	++mNumRetriesTScontent;

				mLastPlaylistFetchTimeUs = -1;
				goto rinse_repeat;
			}else {
			    mDataSource->queueEOS(err);
			}
		}
		else {
        mDataSource->queueEOS(err);
		}
#else
        mDataSource->queueEOS(err);
#endif
        return;
    }

#ifdef SEC_MMFW_HLS
	mNumRetriesTSconnection = 0;
	mNumRetriesTScontent = 0; /* WONPIA 20110317 recv Content Error(timeout, etc) */
#endif
    CHECK(buffer != NULL);

    err = decryptBuffer(mSeqNumber - firstSeqNumberInPlaylist, buffer);

    if (err != OK) {
        LOGE("decryptBuffer failed w/ error %d", err);

        mDataSource->queueEOS(err);
        return;
    }

    if (buffer->size() == 0 || buffer->data()[0] != 0x47) {
        // Not a transport stream???
#if AAC_SUPPORT
        LOGE("This doesn't look like a transport stream...");
	 mAudioOnly = true;
 	 mCheckFirstBW = true;
#else
        mBandwidthItems.removeAt(bandwidthIndex);

        if (mBandwidthItems.isEmpty()) {
            mDataSource->queueEOS(ERROR_UNSUPPORTED);
            return;
        }

        LOGI("Retrying with a different bandwidth stream.");

        mLastPlaylistFetchTimeUs = -1;
#ifdef SEC_MMFW_HLS
		LOGE("Lowest BW is only Audio");
		mAudioOnly = true;
		mCheckFirstBW = true;
#endif
        bandwidthIndex = getBandwidthIndex();
        mPrevBandwidthIndex = bandwidthIndex;
        mSeqNumber = -1;

        goto rinse_repeat;
#endif		
    }

#ifdef SEC_MMFW_HLS
		//change the BW after read the first TS
	if(!mCheckFirstBW){
		LOGE("Change the BW to selecting High BW");
		bandwidthIndex = getBandwidthIndex();
		mCheckFirstBW = true;
		goto rinse_repeat;
	}
#endif

    if ((size_t)mPrevBandwidthIndex != bandwidthIndex) {
        bandwidthChanged = true;
    }

    if (mPrevBandwidthIndex < 0) {
        // Don't signal a bandwidth change at the very beginning of
        // playback.
        bandwidthChanged = false;
    }
#ifdef SEC_MMFW_HLS
    //Check resolution (hxw)
    //if use the bandwidthChanged 
    LOGE("bandwidthChanged = %d, mDownloadMasterlist = %d, mAudioOnly = %d", bandwidthChanged, mDownloadMasterlist, mAudioOnly);
    if ((bandwidthChanged || mDownloadMasterlist || mAudioOnly) && !mSingleBandwidth)
    {
        int findwidth = 0, findheight = 0;
        
        LOGE("Entering Resolution check");
        parseResolution(buffer, &findwidth, &findheight);
		mBandwidthItems.editItemAt(bandwidthIndex).mBandwidthWidth = findwidth;
		mBandwidthItems.editItemAt(bandwidthIndex).mBandwidthHeight = findheight;
		LOGE("Bandwidth[%d] : Found mBandwidthWidth = %d, mBandwidthHeight = %d",bandwidthIndex, mBandwidthItems.editItemAt(bandwidthIndex).mBandwidthWidth,mBandwidthItems.editItemAt(bandwidthIndex).mBandwidthHeight);

//TODO : Can't Change the Resolution, 
		if(mDownloadMasterlist || mAudioOnly){
			mPrevHeight = findheight;
			mPrevWidth = findwidth;
			LOGE("bandwidthIndex[%d] : mPrevHeight = %d, mPrevWidth =%d", bandwidthIndex, mPrevHeight, mPrevWidth); 
			mAudioOnly = false;
			mDownloadMasterlist = false;			
		}
		else{
			if(bandwidthIndex != mPrevBandwidthIndex){		
//Resolution check, Don't need in P5 			
				if((mPrevHeight != findheight) || (mPrevWidth != findwidth)){
					mPrevHeight = findheight;
					mPrevWidth = findwidth;
					LOGE("BW Chanege with initialize the Decoder becuase of resuolution");
					//mResolution= true;
					//goto rinse_repeat;
				}
				else{
					LOGE("BW Chanege without initialize the Decoder");
					//render initial value
					explicitDiscontinuity = false;
					bandwidthChanged = false;					
				}
			}
			else{
				LOGE("Bandwidth not change Current = %d, Prev = %d", bandwidthIndex, mPrevBandwidthIndex);
			}
		}    	
    }
#endif

    if (explicitDiscontinuity || bandwidthChanged) {
        // Signal discontinuity.

        LOGI("queueing discontinuity (explicit=%d, bandwidthChanged=%d)",
             explicitDiscontinuity, bandwidthChanged);

        sp<ABuffer> tmp = new ABuffer(188);
        memset(tmp->data(), 0, tmp->size());
        tmp->data()[1] = bandwidthChanged;

        mDataSource->queueBuffer(tmp);
    }

    mDataSource->queueBuffer(buffer);

    mPrevBandwidthIndex = bandwidthIndex;
    ++mSeqNumber;

    postMonitorQueue();
}

void LiveSession::onMonitorQueue() {
    if (mSeekTimeUs >= 0
            || mDataSource->countQueuedBuffers() < kMaxNumQueuedFragments) {
        onDownloadNext();
    } else {
//        postMonitorQueue(1000000ll);
        postMonitorQueue(500000ll);
		LOGE("Buffer is full");
    }
}

status_t LiveSession::decryptBuffer(
        size_t playlistIndex, const sp<ABuffer> &buffer) {
    sp<AMessage> itemMeta;
    bool found = false;
    AString method;

    for (ssize_t i = playlistIndex; i >= 0; --i) {
        AString uri;
        CHECK(mPlaylist->itemAt(i, &uri, &itemMeta));

        if (itemMeta->findString("cipher-method", &method)) {
            found = true;
            break;
        }
    }

    if (!found) {
        method = "NONE";
    }

    if (method == "NONE") {
        return OK;
    } else if (!(method == "AES-128")) {
        LOGE("Unsupported cipher method '%s'", method.c_str());
        return ERROR_UNSUPPORTED;
    }

    AString keyURI;
    if (!itemMeta->findString("cipher-uri", &keyURI)) {
        LOGE("Missing key uri");
        return ERROR_MALFORMED;
    }

    ssize_t index = mAESKeyForURI.indexOfKey(keyURI);

    sp<ABuffer> key;
    if (index >= 0) {
        key = mAESKeyForURI.valueAt(index);
    } else {
        key = new ABuffer(16);

        sp<NuHTTPDataSource> keySource = new NuHTTPDataSource;
        status_t err = keySource->connect(keyURI.c_str());

        if (err == OK) {
            size_t offset = 0;
            while (offset < 16) {
                ssize_t n = keySource->readAt(
                        offset, key->data() + offset, 16 - offset);
                if (n <= 0) {
                    err = ERROR_IO;
                    break;
                }

                offset += n;
            }
        }

        if (err != OK) {
            LOGE("failed to fetch cipher key from '%s'.", keyURI.c_str());
            return ERROR_IO;
        }

        mAESKeyForURI.add(keyURI, key);
    }

    AES_KEY aes_key;
    if (AES_set_decrypt_key(key->data(), 128, &aes_key) != 0) {
        LOGE("failed to set AES decryption key.");
        return UNKNOWN_ERROR;
    }

    unsigned char aes_ivec[16];

    AString iv;
    if (itemMeta->findString("cipher-iv", &iv)) {
        if ((!iv.startsWith("0x") && !iv.startsWith("0X"))
                || iv.size() != 16 * 2 + 2) {
            LOGE("malformed cipher IV '%s'.", iv.c_str());
            return ERROR_MALFORMED;
        }

        memset(aes_ivec, 0, sizeof(aes_ivec));
        for (size_t i = 0; i < 16; ++i) {
            char c1 = tolower(iv.c_str()[2 + 2 * i]);
            char c2 = tolower(iv.c_str()[3 + 2 * i]);
            if (!isxdigit(c1) || !isxdigit(c2)) {
                LOGE("malformed cipher IV '%s'.", iv.c_str());
                return ERROR_MALFORMED;
            }
            uint8_t nibble1 = isdigit(c1) ? c1 - '0' : c1 - 'a' + 10;
            uint8_t nibble2 = isdigit(c2) ? c2 - '0' : c2 - 'a' + 10;

            aes_ivec[i] = nibble1 << 4 | nibble2;
        }
    } else {
        memset(aes_ivec, 0, sizeof(aes_ivec));
        aes_ivec[15] = mSeqNumber & 0xff;
        aes_ivec[14] = (mSeqNumber >> 8) & 0xff;
        aes_ivec[13] = (mSeqNumber >> 16) & 0xff;
        aes_ivec[12] = (mSeqNumber >> 24) & 0xff;
    }

    AES_cbc_encrypt(
            buffer->data(), buffer->data(), buffer->size(),
            &aes_key, aes_ivec, AES_DECRYPT);

    // hexdump(buffer->data(), buffer->size());

    size_t n = buffer->size();
    CHECK_GT(n, 0u);

    size_t pad = buffer->data()[n - 1];

    CHECK_GT(pad, 0u);
    CHECK_LE(pad, 16u);
    CHECK_GE((size_t)n, pad);
    for (size_t i = 0; i < pad; ++i) {
        CHECK_EQ((unsigned)buffer->data()[n - 1 - i], pad);
    }

    n -= pad;

    buffer->setRange(buffer->offset(), n);

    return OK;
}

void LiveSession::postMonitorQueue(int64_t delayUs) {
    sp<AMessage> msg = new AMessage(kWhatMonitorQueue, id());
    msg->setInt32("generation", ++mMonitorQueueGeneration);
    msg->post(delayUs);
}

void LiveSession::onSeek(const sp<AMessage> &msg) {
    int64_t timeUs;
    CHECK(msg->findInt64("timeUs", &timeUs));

    mSeekTimeUs = timeUs;
    postMonitorQueue();
}

status_t LiveSession::getDuration(int64_t *durationUs) {
    Mutex::Autolock autoLock(mLock);
    *durationUs = mDurationUs;

    return OK;
}

bool LiveSession::isSeekable() {
    int64_t durationUs;
    return getDuration(&durationUs) == OK && durationUs >= 0;
}

#ifdef SEC_MMFW_HLS
// Check for Resolution
void LiveSession::getProgramID(ABitReader *br, unsigned *selProgram) {

	Vector<unsigned> mPrograms;

	unsigned table_id = br->getBits(8);
	LOGV("  table_id = %u", table_id);
	CHECK_EQ(table_id, 0x00u);

	unsigned section_syntax_indictor = br->getBits(1);
	LOGV("  section_syntax_indictor = %u", section_syntax_indictor);
	CHECK_EQ(section_syntax_indictor, 1u);

	CHECK_EQ(br->getBits(1), 0u);
	br->skipBits(2);//reserved

	unsigned section_length = br->getBits(12);
	LOGV("  section_length = %u", section_length);
	CHECK((section_length & 0xc00) == 0);

	br->skipBits(16);//transport_stream_id
	br->skipBits(2); //reserved
	br->skipBits(5); //version_number 
	br->skipBits(1); //current_next_indicator
	br->skipBits(8); //section_number
	br->skipBits(8); //last_section_number

	size_t numProgramBytes = (section_length - 5 /* header */ - 4 /* crc */);
	CHECK_EQ((numProgramBytes % 4), 0u);

	for (size_t i = 0; i < numProgramBytes / 4; ++i) {
		unsigned program_number = br->getBits(16);
		//LOGV("    program_number = %u", program_number);

		br->skipBits(3); //reserved

		if (program_number == 0) {
		    br->skipBits(13);//network_PID
		} else {
		    unsigned programMapPID = br->getBits(13);
		    LOGV("    program_map_PID = 0x%04x", programMapPID);
		    mPrograms.push(programMapPID);
		}
	}

	/////// Select One program
	*selProgram = mPrograms.editItemAt(0); //Select first program    
	return;
}

void LiveSession::getStreamID(ABitReader *br, unsigned *selStream) {

	unsigned table_id = br->getBits(8);
	LOGV("  table_id = %u", table_id);
	CHECK_EQ(table_id, 0x02u);

	unsigned section_syntax_indicator = br->getBits(1);
	LOGV("  section_syntax_indicator = %u", section_syntax_indicator);
	CHECK_EQ(section_syntax_indicator, 1u);

	CHECK_EQ(br->getBits(1), 0u);
	br->skipBits(2);

	unsigned section_length = br->getBits(12);
	LOGV("  section_length = %u", section_length);
	CHECK((section_length & 0xc00) == 0);
	CHECK_LE(section_length, 1021u);

	//MY_LOGI("  program_number = %u", br->getBits(16));
	//MY_LOGI("  reserved = %u", br->getBits(2));
	//MY_LOGI("  version_number = %u", br->getBits(5));
	//MY_LOGI("  current_next_indicator = %u", br->getBits(1));
	//MY_LOGI("  section_number = %u", br->getBits(8));
	//MY_LOGI("  last_section_number = %u", br->getBits(8));
	//MY_LOGI("  reserved = %u", br->getBits(3));
	//MY_LOGI("  PCR_PID = 0x%04x", br->getBits(13));
	//MY_LOGI("  reserved = %u", br->getBits(4));
	br->skipBits(60);

	unsigned program_info_length = br->getBits(12);
	LOGV("  program_info_length = %u", program_info_length);
	CHECK((program_info_length & 0xc00) == 0);

	br->skipBits(program_info_length * 8);  // skip descriptors

	// infoBytesRemaining is the number of bytes that make up the
	// variable length section of ES_infos. It does not include the
	// final CRC.
	size_t infoBytesRemaining = section_length - 9 - program_info_length - 4;

	while (infoBytesRemaining > 0) {
	    CHECK_GE(infoBytesRemaining, 5u);

	    unsigned streamType = br->getBits(8);
	    LOGV("    stream_type = 0x%02x", streamType);

	    br->skipBits(3); //reserved

	    unsigned elementaryPID = br->getBits(13);
	    LOGV("    elementary_PID = 0x%04x", elementaryPID);

	    br->skipBits(4); //reserved(4));

	    unsigned ES_info_length = br->getBits(12);
	    LOGV("    ES_info_length = %u", ES_info_length);
	    CHECK((ES_info_length & 0xc00) == 0);

	    CHECK_GE(infoBytesRemaining - 5, ES_info_length);

#if 0
	    br->skipBits(ES_info_length * 8);  // skip descriptors
#else
	    unsigned info_bytes_remaining = ES_info_length;
	    while (info_bytes_remaining >= 2) {
	        br->skipBits(8);//      tag = 0x%02x"

	        unsigned descLength = br->getBits(8);
	        LOGV("      len = %u", descLength);

	        CHECK_GE(info_bytes_remaining, 2 + descLength);

	        br->skipBits(descLength * 8);

	        info_bytes_remaining -= descLength + 2;
	    }
	    CHECK_EQ(info_bytes_remaining, 0u);
#endif

	   // ssize_t index = mStreams.indexOfKey(elementaryPID);
#if 0  // XXX revisit
	    CHECK_LT(index, 0);
	    mStreams.add(elementaryPID,
	                 new Stream(this, elementaryPID, streamType));
#else
	    //if (index < 0) {
	        //mStreams.add(elementaryPID,
	            //         new Stream(this, elementaryPID, streamType));
	     //Added for Res-Parsing
	     if(streamType == 0x1b)
		 *selStream = elementaryPID; //Update Video elementary PID
	    //}
#endif
	    infoBytesRemaining -= 5 + ES_info_length;
	}

	CHECK_EQ(infoBytesRemaining, 0u);
	br->getBits(32); //CRC     

	return;
}

void LiveSession::getPayloadData(ABitReader *br, uint8_t* payload, unsigned *payloadSize) {

    unsigned packet_startcode_prefix = br->getBits(24);
    //LOGI("packet_startcode_prefix = 0x%08x", packet_startcode_prefix);
    CHECK_EQ(packet_startcode_prefix, 0x000001u);

    unsigned stream_id = br->getBits(8);
    //LOGI("stream_id = 0x%02x", stream_id);

    unsigned PES_packet_length = br->getBits(16);
    //LOGI("PES_packet_length = %u", PES_packet_length);

    if (stream_id != 0xbc  // program_stream_map
            && stream_id != 0xbe  // padding_stream
            && stream_id != 0xbf  // private_stream_2
            && stream_id != 0xf0  // ECM
            && stream_id != 0xf1  // EMM
            && stream_id != 0xff  // program_stream_directory
            && stream_id != 0xf2  // DSMCC
            && stream_id != 0xf8) {  // H.222.1 type E
        CHECK_EQ(br->getBits(2), 2u);

        //MY_LOGI("PES_scrambling_control = %u", br->getBits(2));
        //MY_LOGI("PES_priority = %u", br->getBits(1));
        //MY_LOGI("data_alignment_indicator = %u", br->getBits(1));
        //MY_LOGI("copyright = %u", br->getBits(1));
        //MY_LOGI("original_or_copy = %u", br->getBits(1));
        br->skipBits(6);

        unsigned PTS_DTS_flags = br->getBits(2);
        LOGV("PTS_DTS_flags = %u", PTS_DTS_flags);

        unsigned ESCR_flag = br->getBits(1);
        LOGV("ESCR_flag = %u", ESCR_flag);

        unsigned ES_rate_flag = br->getBits(1);
        LOGV("ES_rate_flag = %u", ES_rate_flag);

        unsigned DSM_trick_mode_flag = br->getBits(1);
        LOGV("DSM_trick_mode_flag = %u", DSM_trick_mode_flag);

        unsigned additional_copy_info_flag = br->getBits(1);
        LOGV("additional_copy_info_flag = %u", additional_copy_info_flag);

        //MY_LOGI("PES_CRC_flag = %u", br->getBits(1));
        //MY_LOGI("PES_extension_flag = %u", br->getBits(1));
        br->skipBits(2);

        unsigned PES_header_data_length = br->getBits(8);
        LOGV("PES_header_data_length = %u", PES_header_data_length);

        unsigned optional_bytes_remaining = PES_header_data_length;

        uint64_t PTS = 0, DTS = 0;

        if (PTS_DTS_flags == 2 || PTS_DTS_flags == 3) {
            CHECK_GE(optional_bytes_remaining, 5u);

            CHECK_EQ(br->getBits(4), PTS_DTS_flags);

            PTS = ((uint64_t)br->getBits(3)) << 30;
            CHECK_EQ(br->getBits(1), 1u);
            PTS |= ((uint64_t)br->getBits(15)) << 15;
            CHECK_EQ(br->getBits(1), 1u);
            PTS |= br->getBits(15);
            CHECK_EQ(br->getBits(1), 1u);

            //LOGV("PTS = %llu", PTS);
            //LOGI("PTS = %.2f secs", PTS / 90000.0f);

            optional_bytes_remaining -= 5;

            if (PTS_DTS_flags == 3) {
                CHECK_GE(optional_bytes_remaining, 5u);

                CHECK_EQ(br->getBits(4), 1u);

                DTS = ((uint64_t)br->getBits(3)) << 30;
                CHECK_EQ(br->getBits(1), 1u);
                DTS |= ((uint64_t)br->getBits(15)) << 15;
                CHECK_EQ(br->getBits(1), 1u);
                DTS |= br->getBits(15);
                CHECK_EQ(br->getBits(1), 1u);

                //LOGI("DTS = %llu", DTS);
                //LOGI("DTS = %.2f secs", DTS / 90000.0f);

                optional_bytes_remaining -= 5;
            }
        }

        if (ESCR_flag) {
            CHECK_GE(optional_bytes_remaining, 6u);

            br->getBits(2);

            uint64_t ESCR = ((uint64_t)br->getBits(3)) << 30;
            CHECK_EQ(br->getBits(1), 1u);
            ESCR |= ((uint64_t)br->getBits(15)) << 15;
            CHECK_EQ(br->getBits(1), 1u);
            ESCR |= br->getBits(15);
            CHECK_EQ(br->getBits(1), 1u);

            LOGV("ESCR = %llu", ESCR);
            //MY_LOGI("ESCR_extension = %u", br->getBits(9));
            br->skipBits(9);

            CHECK_EQ(br->getBits(1), 1u);

            optional_bytes_remaining -= 6;
        }

        if (ES_rate_flag) {
            CHECK_GE(optional_bytes_remaining, 3u);

            CHECK_EQ(br->getBits(1), 1u);
            //MY_LOGI("ES_rate = %u", br->getBits(22));
            br->skipBits(22);
            CHECK_EQ(br->getBits(1), 1u);

            optional_bytes_remaining -= 3;
        }

        br->skipBits(optional_bytes_remaining * 8);

        // ES data follows.

        if (PES_packet_length != 0) {
            CHECK_GE(PES_packet_length, PES_header_data_length + 3);

            unsigned dataLength =
                PES_packet_length - 3 - PES_header_data_length;

            CHECK_GE(br->numBitsLeft(), dataLength * 8);
	     
	     size_t payloadSizeBits = br->numBitsLeft();
            CHECK((payloadSizeBits % 8) == 0);
            
            memcpy(payload, br->data(), payloadSizeBits / 8);
            *payloadSize = payloadSizeBits / 8;

	     LOGV("There's %d bytes of payload.", payloadSizeBits / 8);
            br->skipBits(dataLength * 8);
			
        } else {		
            size_t payloadSizeBits = br->numBitsLeft();
            CHECK((payloadSizeBits % 8) == 0);
			
            memcpy(payload, br->data(), payloadSizeBits / 8);
            *payloadSize = payloadSizeBits / 8;
            
            LOGV("There's %d bytes of payload.", payloadSizeBits / 8);
         }
    } else if (stream_id == 0xbe) {  // ssadding_stream
        CHECK_NE(PES_packet_length, 0u);
        br->skipBits(PES_packet_length * 8);
    } else {
        CHECK_NE(PES_packet_length, 0u);
        br->skipBits(PES_packet_length * 8);
    }
    return;
}

bool LiveSession::parseResolution(const sp<ABuffer> &buffer, int* findwidth, int* findheight) {
    int32_t wd = 0;
    int32_t ht = 0;
    int readSize = 0;
    
    bool isVideo = false;
    
    unsigned mSelProgram = 0u;
    unsigned mVideoElementaryPID = 0u;
    bool mPESstarted = false;
    bool mPEScaptured = false;
    sp<ABuffer> mBuffer, mPayload;
	
    mBuffer = (new ABuffer(128 * 1024)); //Populate Video PES in this buffer
    mPayload = (new ABuffer(128 * 1024)); //Populate Video frames in this buffer
    mBuffer->setRange(0,0);
    mPayload->setRange(0,0);

    //FILE *fp1 = fopen("/data/dump_out.264", "ab+");
	

    //Each TS packet is of 188 bytes is parsed
    while (readSize < buffer->size()) {

	 ABitReader br_new((const uint8_t *)buffer->data()+readSize, 188);//android::kTSPacketSize);
	 ABitReader *br = &br_new;       
        
	 // Identify Video -PES
	 unsigned sync_byte = br->getBits(8);
	 CHECK_EQ(sync_byte, 0x47u);

	 br->skipBits(1); //transport error
	 
	 unsigned payload_unit_start_indicator = br->getBits(1);
	 LOGV("payload_unit_start_indicator = %u", payload_unit_start_indicator);

	 br->skipBits(1); //transport priority

	 unsigned PID = br->getBits(13);
	 LOGV("PID = 0x%04x", PID);

        br->skipBits(2); //control bits (scrambling)

        unsigned adaptation_field_control = br->getBits(2);
        LOGV("adaptation_field_control = %u", adaptation_field_control);
        
	 unsigned continuity_counter = br->getBits(4);
	 LOGV("continuity_counter = %u", continuity_counter);

	 if (adaptation_field_control == 2 || adaptation_field_control == 3) {
            unsigned adaptation_field_length = br->getBits(8);
            if (adaptation_field_length > 0) {
                br->skipBits(adaptation_field_length * 8);  // XXX
            }
        }

	 if (adaptation_field_control == 1 || adaptation_field_control == 3) {
	 	//parsePID(br, PID, payload_unit_start_indicator);
 	    if(PID == 0)
 	    {  //check for Program info in PAT
 	        mSelProgram = 0;
		 mVideoElementaryPID = 0;
 	        if(payload_unit_start_indicator)
		 {
		 	unsigned skip = br->getBits(8);
		 	br->skipBits(skip * 8);
		 }
 	        getProgramID(br, &mSelProgram);
 	    }
	    else
	    {
	        //Check for Stream Info in Program Map
	        //for (size_t i = 0; i < mPrograms.size(); ++i)
	        
	        if (PID == mSelProgram) {
			 mVideoElementaryPID = 0;
			 if(payload_unit_start_indicator)
			 {
			 	unsigned skip = br->getBits(8);
			 	br->skipBits(skip * 8);
			 }
			 getStreamID(br, &mVideoElementaryPID);
	        }
		 else if (PID == mVideoElementaryPID) {
		 	 if(payload_unit_start_indicator)
			 {
			      if(mPESstarted == true)
			      {
				    //One (Video)PES captured - Check Resolution wxh
				    	{				
					        ABitReader br1(mBuffer->data(), mBuffer->size());
						 unsigned nSize = 0;

						 mPayload->data()[0] = 0x00; mPayload->data()[1] = 0x00;
						 mPayload->data()[2] = 0x00; mPayload->data()[3] = 0x01; mPayload->data()[4] = 0x02;
					        mPayload->setRange(0,5);
							
						 //fwrite(mBuffer->data(), mBuffer->size(), 1, fp1);
						 getPayloadData(&br1, mPayload->data()+mPayload->size(), &nSize);
		
				               mPayload->setRange(0, mPayload->size() + nSize);
		
			                      //LOGI("Payload[%d] is ready %x %x %x %x", mPayload->size(),mPayload->data()[0], mPayload->data()[1], mPayload->data()[2], mPayload->data()[3]);						 
		
						 if(mPayload->size() > 0) {
		
						     //fwrite(mPayload->data(), mPayload->size(), 1, fp1);
						     size_t stopOffset = 0;
		
						     //Now get height and width - using AVC utils
						     sp<ABuffer> mSPSNALau = FindNAL (mPayload->data(),  mPayload->size(), 7, &stopOffset);
		
						     if(mSPSNALau != NULL){
						         FindAVCDimensions(mSPSNALau, &wd, &ht);
							  //LOGI("***************Width x height = %dx%d", wd, ht);	    	    
							  if(wd!=0 && ht!=0)
							  	break;
						     }
						 }
						 
				    	}
					mBuffer->setRange(0, 0);
		
	      			}
	      			mPESstarted = true;
 	 		}

	
	
			 if(mPESstarted)
		
			 {
		
			       size_t payloadSizeBits = br->numBitsLeft();
		        	CHECK((payloadSizeBits % 8) == 0);

				LOGV("Adding Payload - %d",payloadSizeBits/8);
			    	CHECK_LE(mBuffer->size() + payloadSizeBits / 8, mBuffer->capacity());
			    	memcpy(mBuffer->data() + mBuffer->size(), br->data(), payloadSizeBits / 8);
			    	mBuffer->setRange(0, mBuffer->size() + payloadSizeBits / 8);
		
			 }
		
 		}
  
   	   }      
      }	 
      readSize+=188;//android::kTSPacketSize;
  }   
  
  *findwidth = wd;
  *findheight= ht;
  
  return true;
}
#endif
} // namespace android
