/* GStreamer
 * Copyright (C) 2008 Wim Taymans <wim.taymans at gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <gst/gst.h>
#include <gst/app/app.h>
#include <stdio.h>
#include <stdlib.h>

#include <gst/rtsp-server/rtsp-server.h>

typedef struct
{
  GstElement *generator_pipe;
  GstElement *vid_appsink;
  GstElement *vid_appsrc;
  GThread *video_change_thread;
  GTimer *timer;
  double last_time;
} MyContext;

static gpointer video_change(MyContext * ctx);

static gpointer video_change(MyContext * ctx)
{
  MyContext *custom_data = ctx;
  gboolean running = TRUE;
  static int count = 0;
  double cur_time = 0;
  while(running) {
  	cur_time = g_timer_elapsed(custom_data->timer, NULL);
	if(cur_time - custom_data->last_time > 1) {
		count = count + 2;
                count = count % 130;
		
       g_object_set(ctx->generator_pipe, "change-video", count, NULL);	
       g_print("\n video_change ongoing to ID %d !!",count);

	custom_data->last_time = cur_time;
	}

	if(count > 130) {
		running = FALSE;
	}
  }
  
  return NULL;
}

static void
ctx_free (MyContext * ctx)
{
  gst_element_set_state (ctx->generator_pipe, GST_STATE_NULL);

  g_printerr ("<<<<<<<<<<<<< FREEING RESOURCE >>>>>>>>>>> \n");
  g_thread_join (ctx->video_change_thread);
  ctx->video_change_thread = NULL;
  g_timer_destroy (ctx->timer);
  gst_object_unref (ctx->generator_pipe);
  gst_object_unref (ctx->vid_appsrc);
  gst_object_unref (ctx->vid_appsink);
  

  g_free (ctx);
}

static GstFlowReturn new_sample (GstElement *sink, MyContext *data) {

  GstSample *sample;
  GstFlowReturn ret;

  MyContext *ctx = (MyContext *)data;
  g_print ("new_sample \n");
  GST_DEBUG_BIN_TO_DOT_FILE(GST_BIN(ctx->generator_pipe), GST_DEBUG_GRAPH_SHOW_ALL, "player2.dot");
  /* Retrieve the buffer */
  g_signal_emit_by_name (sink, "pull-sample", &sample);
  if (sample) {
    /* The only thing we do in this example is print a * to indicate a received buffer */
    g_print ("*");

    g_print ("sample found \n");
    GstBuffer *buffer = gst_sample_get_buffer (sample);
    GstSegment *seg = gst_sample_get_segment (sample);
    GstClockTime pts, dts;

    /* Convert the PTS/DTS to running time so they start from 0 */
    pts = GST_BUFFER_PTS (buffer);
    if (GST_CLOCK_TIME_IS_VALID (pts))
      pts = gst_segment_to_running_time (seg, GST_FORMAT_TIME, pts);

    dts = GST_BUFFER_DTS (buffer);
    if (GST_CLOCK_TIME_IS_VALID (dts))
      dts = gst_segment_to_running_time (seg, GST_FORMAT_TIME, dts);

    if (buffer) {
      /* Make writable so we can adjust the timestamps */
      buffer = gst_buffer_copy (buffer);
      GST_BUFFER_PTS (buffer) = pts;
      GST_BUFFER_DTS (buffer) = dts;
      g_print ("push-buffer !\n");
      g_signal_emit_by_name (data->vid_appsrc, "push-buffer", buffer, &ret);
    }

    /* we don't need the appsink sample anymore */
    gst_sample_unref (sample);
    return GST_FLOW_OK;
  } else {
    g_print ("sample is null!\n");
    return GST_FLOW_ERROR;
  }
}

/* called when a new media pipeline is constructed. We can query the
 * pipeline and configure our appsrc */
static void
media_configure (GstRTSPMediaFactory * factory, GstRTSPMedia * media,
    gpointer user_data)
{
  GstCaps *caps;
  MyContext *ctx;
  GstElement *element, *bin, *encoder_h264, *myappsink, *myappsrc , *fakesink , *vconvert;
  GstPad *pad, *ghost_pad;
  gboolean bRet = FALSE;

  g_print ("media_configure\n");

  ctx = g_new0 (MyContext, 1);
  /* Build the pipeline */
  g_print ("creating pipeline\n");
  ctx->generator_pipe = gst_parse_launch ("playbin3 uri=file:///root/gstreamer/contents/HD.mp4", NULL);

  /* Create the elements inside the sink bin */
  g_print ("creating encoder!\n");
  encoder_h264= gst_element_factory_make ("x264enc", "x264enc");

  g_print ("creating appsink\n");
  myappsink = gst_element_factory_make ("appsink", "appsink");

  g_print ("creating fakesink for audio\n");
  fakesink = gst_element_factory_make ("fakesink", "fakesink");

  g_print ("creating vconv before video encoder \n");
  vconvert = gst_element_factory_make ("videoconvert", "videoconvert");

  if (!encoder_h264 || !myappsink || !vconvert) {
    g_printerr ("Not all elements could be created.\n");
    return ;
  }

  g_object_set (myappsink, "emit-signals", TRUE, NULL);
  g_object_set (myappsink, "max-buffers", 1, NULL);
  g_object_set (myappsink, "drop", TRUE, NULL);
  g_object_set (encoder_h264, "speed-preset", 0x00000002, NULL);
  g_object_set (encoder_h264, "tune", 0x00000002, NULL);

  /* Create the sink bin, add the elements and link them */
  g_print ("create video sink bin\n");
  bin = gst_bin_new ("video_sink_bin\n");
  gst_bin_add_many (GST_BIN (bin), vconvert , encoder_h264, myappsink, NULL);

  g_print ("link video sink bin elements \n");
  bRet = gst_element_link_many (vconvert , encoder_h264, myappsink, NULL);
  g_print ("linking done = %d\n",bRet);
  pad = gst_element_get_static_pad (vconvert, "sink");
  ghost_pad = gst_ghost_pad_new ("sink", pad);
  
  g_print ("set ghost pad active\n");
  gst_pad_set_active (ghost_pad, TRUE);
  gst_element_add_pad (bin, ghost_pad);
  gst_object_unref (pad);

  g_print ("set new-sample signal! \n");
  g_signal_connect (myappsink, "new-sample", G_CALLBACK (new_sample), ctx);

  g_print ("set video-sink \n");
  g_object_set (GST_OBJECT (ctx->generator_pipe), "video-sink", bin, NULL);
  g_object_set (GST_OBJECT (ctx->generator_pipe), "audio-sink", fakesink, NULL);

  ctx->video_change_thread = g_thread_new ("video loop", (GThreadFunc) video_change, ctx);
  ctx->timer = g_timer_new ();
  g_timer_start (ctx->timer);
  ctx->last_time = g_timer_elapsed(ctx->timer, NULL);
  g_print ("Thread and Timer Created !!!!!!!!!!!!!!!!");
  

  g_print ("set pipeline to playing state! \n");
  gst_element_set_state (ctx->generator_pipe, GST_STATE_PLAYING);

  /* make sure the data is freed when the media is gone */
  g_print ("set rtsp-extra-data \n");
  g_object_set_data_full (G_OBJECT (media), "rtsp-extra-data", ctx,
      (GDestroyNotify) ctx_free);

  /* get the element (bin) used for providing the streams of the media */
  element = gst_rtsp_media_get_element (media);

  /* Find the 2 app sources (video / audio), and configure them, connect to the
   * signals to request data */
  /* configure the caps of the video */
  caps = gst_caps_new_simple ("video/x-h264",
      "stream-format", G_TYPE_STRING, "byte-stream",
      "alignment", G_TYPE_STRING, "au",
      "width", G_TYPE_INT, 1280, "height", G_TYPE_INT, 768,
      "framerate", GST_TYPE_FRACTION, 30, 1, NULL);

  //g_print ("get videosrc \n");
  ctx->vid_appsrc = myappsrc =
      gst_bin_get_by_name_recurse_up (GST_BIN (element), "videosrc");

  g_print ("get myappsink \n");
  ctx->vid_appsink = myappsink;

  if(!myappsrc) {
    g_print ("myappsrc is null! \n");
  }
  else
    g_print ("set myappsrc caps \n");

  g_object_set (G_OBJECT (myappsrc), "caps", caps, NULL);

  if(!myappsink) {
  	  g_print ("myappsink is null! \n");
  
}

  gst_object_unref (element);
  g_print ("media_configure done! \n");

  GST_DEBUG_BIN_TO_DOT_FILE(GST_BIN(ctx->generator_pipe), GST_DEBUG_GRAPH_SHOW_ALL, "player.dot");
}

int
main (int argc, char *argv[])
{
  GMainLoop *loop;
  GstRTSPServer *server;
  GstRTSPMountPoints *mounts;
  GstRTSPMediaFactory *factory;

  gst_init (&argc, &argv);

  loop = g_main_loop_new (NULL, FALSE);

  /* create a server instance */
  server = gst_rtsp_server_new ();

  /* get the mount points for this server, every server has a default object
   * that be used to map uri mount points to media factories */
  mounts = gst_rtsp_server_get_mount_points (server);

  /* make a media factory for a test stream. The default media factory can use
   * gst-launch syntax to create pipelines.
   * any launch line works as long as it contains elements named pay%d. Each
   * element with pay%d names will be a stream */
  factory = gst_rtsp_media_factory_new ();

  gst_rtsp_media_factory_set_launch (factory,
      "( appsrc name=videosrc ! queue ! h264parse config-interval=-1 ! queue ! rtph264pay name=pay0 pt=96 )");
  
  /* notify when our media is ready, This is called whenever someone asks for
   * the media and a new pipeline with our appsrc is created */
  g_signal_connect (factory, "media-configure", (GCallback) media_configure,
      NULL);

  /* attach the test factory to the /test url */
  gst_rtsp_mount_points_add_factory (mounts, "/test", factory);

  /* don't need the ref to the mounts anymore */
  g_object_unref (mounts);

  /* attach the server to the default maincontext */
  gst_rtsp_server_attach (server, NULL);

  /* start serving */
  g_print ("stream ready at rtsp://127.0.0.1:8554/test\n");
  g_main_loop_run (loop);

  return 0;
}
