#include "media_common.h"
#include "libavformat/avformat.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <errno.h>

#define BUFF_LEN 128

const char* mime_type_mp4 = "video/mp4";
const char* mime_type_quicktime = "video/quicktime";
const char* mime_type_flv = "video/x-flv";
const char* mime_type_wmv = "video/x-ms-wmv";
const char* mime_type_matroska = "video/x-matroska";
const char* mime_type_mpegts = "video/mp2t";
const char* mime_type_unknown = "application/octet-stream";

int get_buffer_from_file(char* filename, char* buffer, int buff_len);
int add_video_stream_context(MediaContext* context, AVStream* st);
int add_audio_stream_context(MediaContext* context, AVStream* st);
int set_video_stream_context(VideoStreamContext* video, AVStream* st);
int set_audio_stream_context(AudioStreamContext* audio, AVStream* st);
void free_video_stream_context(VideoStreamContext* video_stream_context);
void free_audio_stream_context(AudioStreamContext* audio_stream_context);
void free_subtitle_stream_context(SubtitleStreamContext* subtitle_stream_context);

void media_context_register(void)
{
	av_register_all();
	return;
}

MediaContext* create_media_context(void)
{
	MediaContext* context = NULL;

	errno = 0;
	context = (MediaContext*)calloc(sizeof(MediaContext), 1);
	if ( context == NULL )
	{
		fprintf(stderr, "[%s]error[%s] \n", __func__, strerror(errno));
		return NULL;
	}

	context->filename = NULL;
	context->container_name = NULL;
	context->duration = 0;

	context->video = NULL;
	context->audio = NULL;
	context->subtitle = NULL;

	return context;
}

int build_media_info(MediaContext* context, char* filename)
{
	AVFormatContext* ctx = NULL;
	int ret = 0;
	int i = 0;
	FILE* fp;

	if ( context == NULL )
	{
		return -1;
	}

	errno = 0;
	fp = fopen(filename, "rb");
	if ( fp == NULL )
	{
		return -2;
	}

	fseek(fp, 0, SEEK_END);
	context->file_size = ftell(fp);

	fclose(fp);

	ret = avformat_open_input(&ctx, filename, NULL, NULL);
	if ( ret < 0)
	{
		return 1;
	}

	context->filename = (char*)calloc(strlen(filename) + 1, 1);
	strcpy(context->filename, filename);

	context->container_name = (char*)calloc(strlen(ctx->iformat->name) + 1, 1);
	strcpy(context->container_name, ctx->iformat->name);

	context->duration = (int)ctx->duration/1000;
	context->bit_rate = (int)ctx->bit_rate;

	// set mime type
	if ( strstr(context->container_name, "mov") != NULL )
	{

		AVDictionaryEntry *rot = av_dict_get(ctx->metadata, "major_brand", NULL, 0);
		if ( rot && rot->value )
		{
			if ( strcmp(rot->value, "qt  ") == 0 )
			{
				context->mime_type = INTERNET_MEDIA_TYPE_VIDEO_QUICKTIME;
				context->mime_type_str = (char*)calloc(strlen(mime_type_quicktime) + 1, 1);
				strcpy(context->mime_type_str, mime_type_quicktime);
			}
			else
			{
				context->mime_type = INTERNET_MEDIA_TYPE_VIDEO_MP4;
				context->mime_type_str = (char*)calloc(strlen(mime_type_mp4) + 1, 1);
				strcpy(context->mime_type_str, mime_type_mp4);
			}
		}
		else
		{
			context->mime_type = INTERNET_MEDIA_TYPE_VIDEO_MP4;
			context->mime_type_str = (char*)calloc(strlen(mime_type_mp4) + 1, 1);
			strcpy(context->mime_type_str, mime_type_mp4);
		}
	}
	else if ( strstr(context->container_name, "flv") != NULL )
	{
		context->mime_type = INTERNET_MEDIA_TYPE_VIDEO_X_FLV;
		context->mime_type_str = (char*)calloc(strlen(mime_type_flv) + 1, 1);
		strcpy(context->mime_type_str, mime_type_flv);
	}
	else if ( strstr(context->container_name, "matroska") != NULL )
	{
		context->mime_type = INTERNET_MEDIA_TYPE_VIDEO_X_MATROSKA;
		context->mime_type_str = (char*)calloc(strlen(mime_type_matroska) + 1, 1);
		strcpy(context->mime_type_str, mime_type_matroska);
	}
	else if ( strstr(context->container_name, "asf") != NULL )
	{
		context->mime_type = INTERNET_MEDIA_TYPE_VIDEO_X_MS_WMV;
		context->mime_type_str = (char*)calloc(strlen(mime_type_wmv) + 1, 1);
		strcpy(context->mime_type_str, mime_type_wmv); 
	}
	else if ( strstr(context->container_name, "mpegts") != NULL )
	{
		context->mime_type = INTERNET_MEDIA_TYPE_VIDEO_MPEGTS;
		context->mime_type_str = (char*)calloc(strlen(mime_type_mpegts) + 1, 1);
		strcpy(context->mime_type_str, mime_type_mpegts); 
	}
	else
	{
		context->mime_type = INTERNET_MEDIA_TYPE_UNKNOWN;
		context->mime_type_str = (char*)calloc(strlen(mime_type_unknown) + 1, 1);
		strcpy(context->mime_type_str, mime_type_unknown);
	}

	ret = avformat_find_stream_info(ctx, NULL);
	if ( ret < 0 )
	{
		avformat_close_input(&ctx);
		return 1;
	}

	for ( i = 0 ; i < ctx->nb_streams ; i ++ )
	{
		AVStream* st = ctx->streams[i];

		if ( st->codec->codec_type == AVMEDIA_TYPE_VIDEO )
		{

			ret = add_video_stream_context(context, st);
			if ( ret != 0 )
			{
				continue;
			}
		}
		else if ( st->codec->codec_type == AVMEDIA_TYPE_AUDIO )
		{

			ret = add_audio_stream_context(context, st);
			if ( ret != 0 )
			{
				continue;
			}
		}
		else if ( st->codec->codec_type == AVMEDIA_TYPE_SUBTITLE)
		{
			printf("================== \n");
			printf("codec type subtitle \n");
		}
		else
		{
			printf("================== \n");
			printf("codec type unknown \n");
		}
	}

	avformat_close_input(&ctx);

	return 0;
}

int free_media_context(MediaContext* context)
{
	if (context == NULL)
		return -256;

	free(context->filename);
	context->filename = NULL;

	free(context->container_name);
	context->container_name = NULL;

	free(context->mime_type_str);
	context->mime_type_str = NULL;

	if ( context->video != NULL )
	{
		free_video_stream_context(context->video);
	}

	if ( context->audio != NULL )
	{
		free_audio_stream_context(context->audio);
	}

	if ( context->subtitle != NULL )
	{
		free_subtitle_stream_context(context->subtitle);
	}

	free(context);

	return 0;
}

int get_buffer_from_file(char* filename, char* buffer, int buff_len)
{
	FILE* fp;
	
	errno = 0;
	fp = fopen(filename, "rb");
	if ( fp == NULL )
	{
		return errno;
	}

	fread(buffer, buff_len, sizeof(char), fp);

	fclose(fp);

	return 0;
}

int add_video_stream_context(MediaContext* context, AVStream* st)
{
	VideoStreamContext* head = context->video;
	VideoStreamContext* curr = head;
	VideoStreamContext* prev = NULL;

	while ( curr != NULL )
	{
		prev = curr;
		curr = curr->next;
	}

	curr = (VideoStreamContext*)calloc(sizeof(VideoStreamContext), 1);

	if ( prev == NULL )
		context->video = curr;
	else
		prev->next = curr;

	return set_video_stream_context(curr, st);
}

int add_audio_stream_context(MediaContext* context, AVStream* st)
{
	AudioStreamContext* head = context->audio;
	AudioStreamContext* curr = head;
	AudioStreamContext* prev = NULL;

	while ( curr != NULL )
	{
		prev = curr;
		curr = curr->next;
	}

	curr = (AudioStreamContext*)calloc(sizeof(AudioStreamContext), 1);

	if ( prev == NULL )
		context->audio = curr;
	else
		prev->next = curr;

	return	set_audio_stream_context(curr, st);
}

int set_video_stream_context(VideoStreamContext* video, AVStream* st)
{
	AVCodec* codec = NULL;
	double fps = 0.0;
	int i = 0;

	if (video == NULL)
	{
		return -1;
	}

	fps = (double)(st->nb_frames * ((double)((int64_t)st->time_base.den) / (int64_t)st->time_base.num) / st->duration);

	if ( fps == (double)0.0 )
	{
		fps = (double)(st->r_frame_rate.num / st->r_frame_rate.den);
	}

	video->fps = fps;

	video->width = st->codec->width;
	video->height = st->codec->height;
	video->bit_rate = st->codec->bit_rate;

	video->duration = (int)st->duration;

	// for H.264
	video->spspps_len = st->codec->extradata_size;
	video->spspps = (uint8_t*)calloc(st->codec->extradata_size, 1);
	if ( video->spspps == NULL )
	{
		return -1;
	}

	memcpy((char*)video->spspps, st->codec->extradata, st->codec->extradata_size);

	codec = avcodec_find_decoder(st->codec->codec_id);
	if (codec == NULL)
	{
		return -1;
	}

	video->codec_name = (char*)calloc(strlen(codec->name) + 1, 1);
	strcpy(video->codec_name, codec->name);

	if ( st->nb_index_entries > 0 )
	{
		video->entry_count = st->nb_index_entries;
		video->entries = (StreamEntry*)calloc(sizeof(StreamEntry) * video->entry_count, 1);

		for ( i = 0 ; i < st->nb_index_entries ; i ++ )
		{
			video->entries[i].pos = st->index_entries[i].pos;
			video->entries[i].timestamp = st->index_entries[i].timestamp;
			video->entries[i].flags = st->index_entries[i].flags;
			video->entries[i].size = st->index_entries[i].size;
			video->entries[i].min_distance= st->index_entries[i].min_distance;
		}
	}

	video->next = NULL;

	return 0;
}

int set_audio_stream_context(AudioStreamContext* audio, AVStream* st)
{
	AVCodec* codec = NULL;
	int i = 0;

	audio->bit_rate = st->codec->bit_rate;
	audio->sample_rate = st->codec->sample_rate;
	audio->channels = st->codec->channels;

	audio->duration = (int)st->duration;

	codec = avcodec_find_decoder(st->codec->codec_id);
	if (codec == NULL)
	{
		return -1;
	}

	audio->codec_name = (char*)calloc(strlen(codec->name) + 1, 1);
	strcpy(audio->codec_name, codec->name);

	if ( st->nb_index_entries > 0 )
	{
		audio->entry_count = st->nb_index_entries;
		audio->entries = (StreamEntry*)calloc(sizeof(StreamEntry) * audio->entry_count, 1);

		for ( i = 0 ; i < st->nb_index_entries ; i ++ )
		{
			audio->entries[i].pos = st->index_entries[i].pos;
			audio->entries[i].timestamp = st->index_entries[i].timestamp;
			audio->entries[i].flags = st->index_entries[i].flags;
			audio->entries[i].size = st->index_entries[i].size;
			audio->entries[i].min_distance= st->index_entries[i].min_distance;
		}
	}

	audio->next = NULL;

	return 0;
}


void free_video_stream_context(VideoStreamContext* video_stream_context)
{
	VideoStreamContext* head = video_stream_context;
	VideoStreamContext* tmp = head;

	while ( head != NULL )
	{
		tmp = head;
		head = head->next;

		free(tmp->codec_name);
		free(tmp->entries);

		free(tmp->spspps);

		free(tmp);

		tmp = NULL;
	}
}

void free_audio_stream_context(AudioStreamContext* audio_stream_context)
{
	AudioStreamContext* head = audio_stream_context;
	AudioStreamContext* tmp = head;

	while ( head != NULL )
	{
		tmp = head;
		head = head->next;

		free(tmp->codec_name);
		free(tmp->entries);

		free(tmp);

		tmp = NULL;
	}
}

void free_subtitle_stream_context(SubtitleStreamContext* subtitle_stream_context)
{
	SubtitleStreamContext* head = subtitle_stream_context;
	SubtitleStreamContext* tmp = head;

	while ( head != NULL )
	{
		tmp = head;
		head = head->next;

		free(tmp->codec_name);
		free(tmp->entries);

		free(tmp);

		tmp = NULL;
	}
}

